from sqlalchemy.ext.declarative import declarative_base, declared_attr
from sqlalchemy import Column, Integer, String, Text, DateTime, Boolean, Date
from sqlalchemy import ForeignKey
from sqlalchemy.orm import relationship, backref

Base = declarative_base()


class OEBaseMixin(object):
    id = Column(Integer, primary_key=True)

    @declared_attr
    def _create_uid(cls):
        return Column('create_uid', Integer, ForeignKey('res_users.id'))

    @declared_attr
    def _write_uid(cls):
        return Column('write_uid', Integer, ForeignKey('res_users.id'))

    create_date = Column(DateTime(timezone=False))
    write_date = Column(DateTime(timezone=False))

    @declared_attr
    def create_uid(cls):
        return relationship(
            'ResUser', foreign_keys=cls.__name__ + '._create_uid')

    @declared_attr
    def write_uid(cls):
        return relationship(
            'ResUser', foreign_keys=cls.__name__ + '._write_uid')


class IrModel(OEBaseMixin, Base):
    __tablename__ = 'ir_model'

    model = Column(String(64))
    name = Column(String(64))
    state = Column(String(16))
    info = Column(Text)

    constraints = relationship(
        "IrModelConstraint",
        backref=backref("irmodel", uselist=False)
    )

    fields = relationship(
        "IrModelFields",
        backref=backref("irmodel", uselist=False)
    )

    relations = relationship(
        "IrModelRelation",
        backref=backref("irmodel", uselist=False)
    )

    def __repr__(self):
        return "<IrModel(model='%s', name='%s', id='%s')>" % (
            self.model,
            self.name,
            self.id,
        )

    @classmethod
    def by_modelname(klass, session, modelname):
        """A helper class method to query the database for a model based on
        its model name.

        @param session: a configured session to use
        @type session: sqlalchemy.orm.session instance

        @param modelname: the XMLID of the requested model
        @type modelname: string
        """
        return session.query(IrModel).filter(
            IrModel.model == modelname
        ).first()


class IrModelData(OEBaseMixin, Base):
    __tablename__ = 'ir_model_data'

    noupdate = Column(Boolean)
    name = Column(String(64))
    date_init = Column(DateTime(timezone=False))
    date_update = Column(DateTime(timezone=False))
    module = Column(String(64))
    model = Column(String(64))
    res_id = Column(Integer)

    def __repr__(self):
        return "<IrModelData(name='%s', id='%s')>" % (self.name, self.id)


class IrModelConstraint(OEBaseMixin, Base):
    __tablename__ = 'ir_model_constraint'

    date_init = Column(DateTime(timezone=False))
    date_update = Column(DateTime(timezone=False))
    module = Column(String(64))
    model = Column(Integer, ForeignKey('ir_model.id'))
    type = Column(String(1))
    name = Column(String(128))

    def __repr__(self):
        return "<IrModelConstraint(name='%s', id='%s')>" % (self.name, self.id)


class IrModelFields(OEBaseMixin, Base):
    __tablename__ = 'ir_model_fields'

    model = Column(String(64))
    model_id = Column(Integer, ForeignKey('ir_model.id'))
    name = Column(String(64))
    relation = Column(String(64))
    select_level = Column(String(4))
    field_description = Column(String(256))
    ttype = Column(String(64))
    state = Column(String(64), nullable=False, default='base')

    def __repr__(self):
        return "<IrModelFields(name='%s', id='%s')>" % (self.name, self.id)


class IrModelRelation(OEBaseMixin, Base):
    __tablename__ = 'ir_model_relation'

    date_init = Column(DateTime(timezone=False))
    date_update = Column(DateTime(timezone=False))
    module = Column(Integer)
    model = Column(Integer, ForeignKey('ir_model.id'))
    name = Column(String(128))

    def __repr__(self):
        return "<IrModelRelation(name='%s', id='%s')>" % (self.name, self.id)


class ResCountry(OEBaseMixin, Base):
    __tablename__ = 'res_country'

    name = Column(String())
    code = Column(String())
    address_format = Column(Text())


class ResCountryState(OEBaseMixin, Base):
    __tablename__ = 'res_country_state'

    _country_id = Column('country_id', ForeignKey(ResCountry.id))
    country = relationship(ResCountry)

    name = Column(String())
    code = Column(String())


class ResUser(OEBaseMixin, Base):
    __tablename__ = 'res_users'

    name = Column(String())


class ResPartner(OEBaseMixin, Base):
    __tablename__ = 'res_partner'

    email = Column(String())
    name = Column(String())


class ResPartnerBank(OEBaseMixin, Base):
    __tablename__ = 'res_partner_bank'

    partner_id = Column(ForeignKey('res_partner.id'))
    partner = relationship('ResPartner', backref='bank_list')


class ResCompany(OEBaseMixin, Base):
    __tablename__ = 'res_company'

    name = Column(String())


class AccountFiscalyear(OEBaseMixin, Base):
    __tablename__ = 'account_fiscalyear'

    code = Column(String())
    name = Column(String())
    date_start = Column(Date())
    date_stop = Column(Date())
